'use client';

import { useParams, useRouter } from 'next/navigation';
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
  CardFooter
} from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Ticket, ArrowLeft, Send, LoaderCircle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Textarea } from '@/components/ui/textarea';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { cn } from '@/lib/utils';
import { useToast } from '@/hooks/use-toast';
import { useDoc, useFirestore, useUser, useMemoFirebase } from '@/firebase';
import { doc, updateDoc, arrayUnion, serverTimestamp } from 'firebase/firestore';
import type { SupportTicket, TicketMessage } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { useState } from 'react';
import { errorEmitter } from '@/firebase/error-emitter';
import { FirestorePermissionError } from '@/firebase/errors';


export default function TicketDetailsPage() {
  const params = useParams();
  const router = useRouter();
  const { user } = useUser();
  const firestore = useFirestore();
  const ticketId = params.ticketId as string;
  const { toast } = useToast();

  const [reply, setReply] = useState('');
  const [isReplying, setIsReplying] = useState(false);
  const [isClosing, setIsClosing] = useState(false);

  const ticketDocRef = useMemoFirebase(
    () => (user ? doc(firestore, 'users', user.uid, 'tickets', ticketId) : null),
    [firestore, user, ticketId]
  );
  
  const { data: ticket, isLoading, error } = useDoc<SupportTicket>(ticketDocRef);
  
  const handleReply = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!reply.trim() || !user || !ticketDocRef) return;
    
    setIsReplying(true);

    const newMessage: TicketMessage = {
        sender: 'user',
        message: reply,
        timestamp: serverTimestamp(),
    };

    try {
        await updateDoc(ticketDocRef, {
            conversation: arrayUnion(newMessage),
            updatedAt: serverTimestamp(),
        });
        setReply('');
        toast({
            title: 'Reply Sent',
            description: 'Your reply has been added to the conversation.',
        });
    } catch (e: any) {
        errorEmitter.emit(
            'permission-error',
            new FirestorePermissionError({
              path: ticketDocRef.path,
              operation: 'update',
              requestResourceData: { conversation: arrayUnion(newMessage) },
            })
        );
    } finally {
        setIsReplying(false);
    }
  };

  const handleCloseTicket = async () => {
    if (!user || !ticketDocRef) return;
    setIsClosing(true);
    try {
        await updateDoc(ticketDocRef, {
            status: 'Closed',
            updatedAt: serverTimestamp(),
        });
        toast({
            title: 'Ticket Closed',
            description: `You have closed the support ticket #${ticketId}.`,
        });
    } catch (e: any) {
        errorEmitter.emit(
            'permission-error',
            new FirestorePermissionError({
              path: ticketDocRef.path,
              operation: 'update',
              requestResourceData: { status: 'Closed' },
            })
        );
    } finally {
        setIsClosing(false);
    }
  }

  const getTimestamp = (ts: any) => {
    if (!ts) return '';
    if (ts?.seconds) { // Firestore Timestamp object
      return new Date(ts.seconds * 1000);
    }
    return new Date(ts); // string date
  }

  if (isLoading) {
      return (
          <div className="p-4 md:p-8 space-y-4">
              <Skeleton className="h-8 w-64" />
              <div className="grid gap-6 lg:grid-cols-3">
                  <div className="lg:col-span-2">
                    <Skeleton className="h-96 w-full" />
                  </div>
                  <Skeleton className="h-64 w-full" />
              </div>
          </div>
      )
  }

  if (error || !ticket) {
    return (
        <div className="p-4 md:p-8">
            <Card>
            <CardContent className="pt-6">
                <Alert variant="destructive">
                <Ticket className="h-4 w-4" />
                <AlertTitle>Ticket Not Found</AlertTitle>
                <AlertDescription>
                    The support ticket you are looking for does not exist or you do not have permission to view it.
                </AlertDescription>
                </Alert>
            </CardContent>
            </Card>
      </div>
    );
  }

  return (
    <div className="p-4 md:p-8">
      <div className="flex items-center mb-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => router.back()}
          className="mr-2"
        >
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <h1 className="text-2xl font-bold">Ticket Details</h1>
      </div>

        <div className="grid gap-6 lg:grid-cols-3">
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle>{ticket.subject}</CardTitle>
                <CardDescription>
                  Conversation history for ticket #{ticket.id}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                 <div
                    className={cn(
                      'flex items-start gap-3',
                       'justify-end'
                    )}
                  >
                    <div
                      className={cn(
                        'rounded-lg p-3 max-w-md',
                        'bg-muted'
                      )}
                    >
                      <p className="text-sm font-semibold mb-1">Initial Query:</p>
                      <p className="text-sm">{ticket.description}</p>
                       <p className="text-xs opacity-70 mt-1 text-right">
                        {getTimestamp(ticket.createdAt).toLocaleString()}
                      </p>
                    </div>
                      <Avatar className="h-8 w-8">
                        <AvatarFallback>U</AvatarFallback>
                      </Avatar>
                  </div>
                {ticket.conversation?.map((message, index) => (
                  <div
                    key={index}
                    className={cn(
                      'flex items-start gap-3',
                      message.sender === 'user' ? 'justify-end' : ''
                    )}
                  >
                    {message.sender === 'admin' && (
                      <Avatar className="h-8 w-8">
                        <AvatarFallback>A</AvatarFallback>
                      </Avatar>
                    )}
                    <div
                      className={cn(
                        'rounded-lg p-3 max-w-md',
                        message.sender === 'user'
                          ? 'bg-primary text-primary-foreground'
                          : 'bg-muted'
                      )}
                    >
                      <p className="text-sm">{message.message}</p>
                       <p className="text-xs opacity-70 mt-1 text-right">
                        {getTimestamp(message.timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                      </p>
                    </div>
                    {message.sender === 'user' && (
                      <Avatar className="h-8 w-8">
                        <AvatarFallback>U</AvatarFallback>
                      </Avatar>
                    )}
                  </div>
                ))}
              </CardContent>
              {ticket.status !== 'Closed' && ticket.status !== 'Resolved' && (
                <CardFooter>
                   <form onSubmit={handleReply} className="w-full flex items-start gap-2">
                      <Textarea 
                        placeholder="Type your reply..." 
                        className="flex-1"
                        value={reply}
                        onChange={(e) => setReply(e.target.value)}
                        disabled={isReplying} />
                      <Button type="submit" size="icon" className="shrink-0" disabled={isReplying || !reply.trim()}>
                          {isReplying ? <LoaderCircle className="h-4 w-4 animate-spin" /> : <Send className="h-4 w-4" />}
                          <span className="sr-only">Send</span>
                      </Button>
                  </form>
                </CardFooter>
              )}
            </Card>
          </div>
          <div className="space-y-6">
             <Card>
                <CardHeader>
                    <CardTitle>Ticket Information</CardTitle>
                </CardHeader>
                <CardContent className="text-sm space-y-3">
                    <div className="flex justify-between">
                        <span className="text-muted-foreground">Status:</span>
                        <Badge>{ticket.status}</Badge>
                    </div>
                    <div className="flex justify-between">
                        <span className="text-muted-foreground">Category:</span>
                        <span>{ticket.issueCategory}</span>
                    </div>
                    <div className="flex justify-between">
                        <span className="text-muted-foreground">Order ID:</span>
                        <span>{ticket.orderId || 'N/A'}</span>
                    </div>
                     <div className="flex justify-between">
                        <span className="text-muted-foreground">Created:</span>
                        <span>{getTimestamp(ticket.createdAt).toLocaleDateString()}</span>
                    </div>
                </CardContent>
                {ticket.status !== 'Closed' && (
                <CardFooter>
                    <Button variant="outline" className="w-full" onClick={handleCloseTicket} disabled={isClosing}>
                       {isClosing && <LoaderCircle className="h-4 w-4 animate-spin mr-2" />}
                        Close Ticket
                    </Button>
                </CardFooter>
                )}
             </Card>
          </div>
        </div>
    </div>
  );
}
